text \<open>
  Friends with Benefits:
  Implementing Foundational Corecursion in Isabelle/HOL

  Jasmin  Blanchette    Inria & Loria Nancy, MPI-INF Saarbrücken
  Aymeric Bouzy         Ecole polytechnique Paris
  Andreas Lochbihler    ETH Zurich
  Andrei  Popescu       Middlesex University
  Dmitriy Traytel       ETH Zurich




  GOAL: Define non-primitively corecursive functions!
\<close>

theory BNF_Corec_Demo imports 
  "~~/src/HOL/Library/BNF_Corec"
  Demo_Preliminaries
begin

section \<open>Streams / infinite sequences\<close>

codatatype 'a stream = SCons (shd: 'a) (stl: "'a stream") (infixr "##" 65)
  for map: smap

declare stream.map_sel [simp]

text \<open>Lazy evaluation for streams with @{method code_simp}.\<close>
lemma SCons_lazy: "x = y \<Longrightarrow> x ## xs = y ## xs" by simp
setup {* Code_Simp.map_ss (Simplifier.add_cong @{thm SCons_lazy}) *}

fun stake :: "nat \<Rightarrow> 'a stream \<Rightarrow> 'a list" where
  "stake 0       _         = []"
| "stake (Suc n) (x ## xs) = x # stake n xs"


section \<open>Primitive corecursion\<close>

text \<open>
  primitive recursion              primitive corecursion
  - datatype as argument           - codatatype as result
  - peel off one constructor       - produce one constructor
  - recursive calls only on        - corecursive call only as
    arguments of the constructor     arguments to the constructor\<close>

primrec len :: "'a list \<Rightarrow> nat" where
  "len []       = 0"
| "len (_ # xs) = Suc (len xs)"

primcorec sconst :: "'a \<Rightarrow> 'a stream" where
  "sconst x = x ## sconst x"

value [simp] "sconst 5 :: nat stream"
value [simp] "stake 7 (sconst 5 :: nat stream)"

thm len_def sconst_def


subsection \<open>\<open>1 ## 2 ## 3 ## 4 ## \<dots>\<close>\<close>

text \<open>
  Primitive corecursion for streams:
  \<^item> one single loop
  \<^item> output one element at a time.

  while (true) do {
    output n;
    n++;
  }
\<close>

primcorec up :: "nat \<Rightarrow> nat stream"
where "up n = n ## up (n + 1)"

value [simp] "stake 50 (up 1)"


section \<open>Multiple constructors\<close>

subsection \<open>\<open>True ## False ## True ## False ## True ## False ## \<dots>\<close>\<close>
text \<open>
  +---+      True      +---+            while (true) do {
  |   |--------------->|   |              output z;
  | T |                | F |              z := \<not> z;
  |   |<---------------|   |            }
  +---+     False      +---+
\<close>

primcorec alternate :: "bool \<Rightarrow> bool stream"
where "alternate z = z ## alternate (\<not> z)"

abbreviation "TFs \<equiv> alternate True"

value [simp] "stake 10 TFs"



text \<open>
  Multiple outputs in one iteration:
  +--+                                  while (true) do {
  |  |---------------+                    output True;
  |  |<--------------+                    output False;
  +--+    True,True                     }\<close>

corec TFs' :: "bool stream" where
  "TFs' = True ## False ## TFs'"

value [simp] "stake 10 TFs'"

thm TFs'.unique

lemma "TFs = TFs'"
proof(rule TFs'.unique)
  show "TFs = True ## False ## TFs"
    by(rule stream.expand; simp)+
qed




subsection \<open>\<open>y\<^sub>1, x, y\<^sub>2, x, y\<^sub>3, x, y\<^sub>4, x, \<dots>\<close>\<close>

text \<open>Insert \<open>x\<close> after each element in the stream \<open>ys\<close>.\<close>

context fixes x :: 'a begin

corec ssep :: "'a stream \<Rightarrow> 'a stream"
where "ssep ys = (case ys of y ## ys' \<Rightarrow> y ## x ## ssep ys')"

end

value [simp] "stake 20 (ssep x (up 1))"

lemma ssep_sel [simp]:
  "shd (ssep x ys) = shd ys"
  "stl (ssep x ys) = x ## ssep x (stl ys)"
by(subst ssep.code; simp split: stream.split; fail)+

thm stream.coinduct stream.coinduct_upto

thm ssep.unique
lemma 
  "smap f (ssep x ys) = ssep (f x) (smap f ys)"
  (is "?lhs = ?rhs")
proof -
  def lhs \<equiv> ?lhs and rhs \<equiv> ?rhs
  def R \<equiv> "\<lambda>lhs rhs. \<exists>ys. lhs = smap f (ssep x ys) \<and> rhs = ssep (f x) (smap f ys)"
  have "R lhs rhs" unfolding R_def lhs_def rhs_def by blast
  then show "lhs = rhs"
    apply(rule stream.coinduct_upto)
    apply(unfold R_def)
    apply(clarsimp)
    find_theorems intro
    apply(rule stream.cong_SCons)
     apply simp
    apply(rule stream.cong_base)
    apply auto
    done
qed

lemma map_ssep: "smap f (ssep x xs) = ssep (f x) (smap f xs)" (is "?lhs = ?rhs")
by(coinduction arbitrary: xs rule: stream.coinduct_upto)
  (fastforce intro: stream.cong_SCons stream.cong_base)



section \<open>Corecursion with friends\<close>

primcorec prepend :: "'a list \<Rightarrow> 'a stream \<Rightarrow> 'a stream" (infixr "@@" 65)
where
  "xs @@ ys = (case xs of [] \<Rightarrow> ys
                   | x # xs' \<Rightarrow> x ## (xs' @@ ys))"

(*
primrec prepend :: "'a list \<Rightarrow> 'a stream \<Rightarrow> 'a stream" (infixr "@@" 65) where
  "[]       @@ ys = ys"
| "(x # xs) @@ ys = x ## (xs @@ ys)"
*)

lemma prepend_sel [simp]:
  shows shd_prepend: "shd (xs @@ ys) = (if xs \<noteq> [] then hd xs else shd ys)"
  and stl_prepend:   "stl (xs @@ ys) = (if xs \<noteq> [] then tl xs @@ ys else stl ys)"
by(subst prepend.code; simp split: list.split; fail)+

lemma prepend_simps [simp]:
  "[]       @@ ys = ys"
  "(x # xs) @@ ys = x ## (xs @@ ys)"
by(auto intro: stream.expand)




subsection \<open>\<open>cycle xs = xs @@ cycle xs\<close>\<close>

text \<open>
  while (true) do {
    output_list xs;
  }
\<close>

friend_of_corec prepend where
  "xs @@ ys = (case xs of [] \<Rightarrow> shd ys ## stl ys | x # xs' \<Rightarrow> x ## xs' @@ ys)"
apply(simp split: list.split)
apply(transfer_prover')
done


corec cycle :: "'a list \<Rightarrow> 'a stream"
where "cycle xs = hd xs ## tl xs @@ cycle xs"

value [simp] "stake 10 (cycle ''#:.'')"

lemma cycle_sel [simp]:
  "shd (cycle xs) = hd xs"
  "stl (cycle xs) = tl xs @@ cycle xs"
by(subst cycle.code; simp; fail)+




lemma prepend_append [simp]: "(xs @ ys) @@ zs = xs @@ ys @@ zs"
by(induction xs) simp_all

lemma smap_prepend [simp]: "smap f (xs @@ ys) = map f xs @@ smap f ys"
by(coinduction arbitrary: xs rule: stream.coinduct_strong)(auto split: list.split)


thm stream.coinduct_upto
find_theorems stream.congclp

lemma "xs \<noteq> [] \<Longrightarrow> smap f (cycle xs) = cycle (map f xs)"
apply(coinduction rule: stream.coinduct_upto)
apply simp
apply(rule stream.cong_prepend)
 apply simp
apply(rule stream.cong_base)
apply simp
done



section \<open>Mixing recursion and corecursion\<close>

text \<open>
  while (true) do {
    if prime n then output n;
    n++;
  }
\<close>

corecursive primes_from :: "nat \<Rightarrow> nat stream" where
  "primes_from n = 
   (if prime n then n ## primes_from (n + 1)
    else primes_from (n + 1))"
apply(relation "{(n + 1, n) | n :: nat. \<not> prime n}")
apply(rule wf_next_prime)
apply simp
done


abbreviation "primes \<equiv> primes_from 0"

value [simp] "stake 15 primes"







section \<open>Applications and reading\<close>
text \<open>
  \<^item> Many stream examples (e.g., Hamming sequence)
  \<^item> Formal languages as infinite trees (FSCD 2016)
  \<^item> Knuth-Morris-Pratt string matching
  \<^item> Stern-Brocot tree
  \<^item> Breadth-first tree labelling
  \<^item> Stream processors
  \<^item> filter on lazy lists
  \<^item> monads on codatatypes

  Literature
  \<^item> Foundational Extensible Corecursion (ICFP 2015)
  \<^item> Friends with Benefits: Implementing Foundational Corecursion in Proof Assistants
    (submitted)
  \<^item> Tutorial on Nonprimitively Corecursive Definitions (isabelle doc corec)
\<close>

end
